module Mbackup.Lib

open System
open System.Text.RegularExpressions

type Logger() =
  let mutable level = Logger.DEBUG

  static member DEBUG = 10
  static member INFO = 20
  static member WARNING = 30
  static member ERROR = 40
  static member LevelToString level =
    match level with
      | 10 -> "DEBUG"
      | 20 -> "INFO"
      | 30 -> "WARNING"
      | 40 -> "ERROR"
      | _ -> failwith (sprintf "Unknown log level: %d" level)

  member this.Level = level

  member this.LogMaybe level fmt =
    let doAfter s =
      if this.Level <= level then
        let ts = DateTime.Now.ToString("s")
        printf "%s %s %s\n" ts (Logger.LevelToString level) s
      else
        printf ""
    Printf.ksprintf doAfter fmt

  member this.SetLevel level =
    this.Level = level
  member this.Debug fmt = this.LogMaybe Logger.DEBUG fmt
  member this.Info fmt = this.LogMaybe Logger.INFO fmt
  member this.Warning fmt = this.LogMaybe Logger.WARNING fmt
  member this.Error fmt = this.LogMaybe Logger.ERROR fmt

// append string s to list if pred is true
let appendWhen (pred: bool) (lst: string list) (s: string) = if pred then List.append lst [s] else lst

let getEnv (varName: string) = Environment.GetEnvironmentVariable varName

let getEnvDefault (varName: string) (defaultValue: string) =
  let value = Environment.GetEnvironmentVariable varName
  match value with
    | null -> defaultValue
    | "" -> defaultValue
    | _ -> value

// Convert windows path to Mingw64 path.
// Supported windows path: C:\foo, C:/foo, /c/foo
// MingwPath format: /cygdrive/c/foo
let toMingwPath (windowsPath: string) =
  let pattern = Regex("^/([c-zC-Z])/", RegexOptions.None)
  let result =
    if pattern.IsMatch(windowsPath) then
      "/cygdrive" + windowsPath
    else
      let pattern = Regex("^([c-zC-Z]):", RegexOptions.None)
      if pattern.IsMatch(windowsPath) then
        let result = windowsPath.Replace('\\', '/')
        "/cygdrive/" + result.Substring(0, 1).ToLower() + result.Substring(2)
      else
        windowsPath
  result

let ensureDir (path: string) = if path.EndsWith "/" then path else path + "/"
let ensureWinDir (path: string) = if path.EndsWith "\\" then path else path + "\\"
